# Quick Diagnostics for Live Domain Issues

## Access Your Live Server

### Via SSH (Terminal/Command Line)
```bash
# On Windows PowerShell or Mac/Linux Terminal
ssh username@yourdomain.com

# Or with specific port
ssh -p 2222 username@yourdomain.com

# Enter password when prompted
```

### Via cPanel Terminal
1. Log into cPanel
2. Go to **Terminal** (under Development)
3. Use bash commands directly

---

## 1. Check Current Error

### View Error Log
```bash
# Check last 50 lines
tail -50 /home/username/public_html/storage/logs/laravel.log

# Follow live errors (Ctrl+C to stop)
tail -f /home/username/public_html/storage/logs/laravel.log

# Search for specific error
grep "undefined" /home/username/public_html/storage/logs/laravel.log
grep "database" /home/username/public_html/storage/logs/laravel.log
```

### Check PHP Error Log
```bash
# cPanel hosted usually has:
tail -50 /home/username/public_html/error_log

# Or check Apache error log
tail -50 /var/log/apache2/error_log
```

---

## 2. Verify Database Connection

### Test Database Access
```bash
# SSH into server first
cd /home/username/public_html

# Use Laravel Tinker
php artisan tinker

# Then run these commands:
> DB::connection()->getPdo()
# If successful: object(PDO)
# If error: Shows connection problem

> DB::table('business_settings')->count()
# Should return a number (not 0 ideally)

> DB::table('business_settings')->first()
# Should show config data

> exit
```

### Direct MySQL Test
```bash
# If not in Laravel Tinker
mysql -u live_user -p -h localhost
# Enter password

# In MySQL:
USE live_db_name;
SELECT COUNT(*) FROM business_settings;
SELECT * FROM business_settings LIMIT 1;
EXIT;
```

---

## 3. Check Environment File

### Verify .env File Exists
```bash
cd /home/username/public_html
ls -la | grep .env

# Should show: .env (not .env.example)
```

### Check .env Contents (First 15 Lines)
```bash
head -15 .env

# Should show database credentials matching your setup
```

### View Full .env (Carefully!)
```bash
cat .env

# Look for:
# - APP_DEBUG=false (production)
# - APP_ENV=production
# - DB_HOST, DB_DATABASE, DB_USERNAME, DB_PASSWORD
# - APP_URL=https://yourdomain.com
```

---

## 4. Check Database Tables

### List All Tables
```bash
mysql -u live_user -p'your_password' live_db_name
SHOW TABLES;
```

### Verify Key Tables Exist
```bash
mysql -u live_user -p'your_password' live_db_name -e "
SHOW TABLES LIKE '%business_settings%';
SHOW TABLES LIKE '%users%';
SHOW TABLES LIKE '%products%';
"
```

---

## 5. Test PHP & Laravel

### Check PHP Version
```bash
php --version
# Should be 7.4 or higher
```

### Test Laravel Command Works
```bash
cd /home/username/public_html
php artisan --version

# Should show: Laravel Framework X.XX.X
```

### Run Health Check
```bash
php artisan migrate:status
# Should list all migrations with status
```

---

## 6. Check File Permissions

### View Permissions
```bash
cd /home/username/public_html

# Check storage
ls -ld storage
# Should show: drwxrwxr-x (755 or 775)

# Check bootstrap/cache
ls -ld bootstrap/cache
# Should show: drwxrwxr-x

# Check .env file
ls -l .env
# Should show: -rw------- or -rw-r--r--
```

### Fix Permissions If Needed
```bash
cd /home/username/public_html

# Make storage writable
chmod -R 775 storage/
chmod -R 775 bootstrap/cache/

# Verify
ls -ld storage bootstrap/cache
```

---

## 7. Test Web Access

### Check if Site Responds
```bash
# From any computer (not SSH), use curl
curl -I https://yourdomain.com
# Should show:
# HTTP/2 200  (success)
# or
# HTTP/1.1 500  (error - check logs)

# Get full HTML response
curl https://yourdomain.com | head -50
# Should show HTML, not error page
```

### Check Specific Page
```bash
curl -i https://yourdomain.com/api/health
# Or test a specific route
```

---

## 8. Verify Migrations Ran

### List Migration Status
```bash
cd /home/username/public_html
php artisan migrate:status

# All should show "Batch" number (not blank)
# Example:
# 2014_10_12_000000_create_users_table        Batch 1
# 2014_10_12_100000_create_password_resets... Batch 1
```

### Check If business_settings Table Has Data
```bash
mysql -u live_user -p'your_password' live_db_name -e "
SELECT COUNT(*) as total_settings FROM business_settings;
SELECT key_name FROM business_settings LIMIT 5;
"
```

---

## 9. Quick Fix Commands

### If Database Connection Fails
```bash
cd /home/username/public_html

# Ensure config is loaded
php artisan config:clear
php artisan config:cache

# Test again
php artisan tinker
> DB::connection()->getPdo()
```

### If Migrations Not Run
```bash
cd /home/username/public_html

# Run migrations
php artisan migrate --force

# Seed database
php artisan db:seed --force

# Check if tables created
php artisan migrate:status
```

### If Cache Issues
```bash
cd /home/username/public_html

php artisan cache:clear
php artisan view:clear
php artisan route:cache
php artisan config:cache
```

### If Permissions Issues
```bash
cd /home/username/public_html

# Fix all permissions at once
chmod -R 775 storage/ bootstrap/cache/ public/
chown -R $(whoami):www-data storage/ bootstrap/cache/
```

---

## 10. Monitor Live Logs in Real Time

### Watch Errors As They Happen
```bash
cd /home/username/public_html

# In one terminal:
tail -f storage/logs/laravel.log

# In another terminal, test your site:
# - Open browser to yourdomain.com
# - Refresh page
# - Try different pages

# Watch errors appear in real-time
# Press Ctrl+C to stop
```

---

## 11. Test Each Component

### Test Database
```bash
php artisan tinker
> DB::connection()->getPdo()
> exit
```

### Test Cache
```bash
php artisan cache:clear
# Should succeed
```

### Test File Permissions
```bash
touch storage/test.txt
# Should create file successfully
rm storage/test.txt
```

### Test Laravel Routes
```bash
php artisan route:list | head -20
# Should show all routes
```

---

## 12. Common Issues Quick Fix

| Issue | Check | Command |
|-------|-------|---------|
| 500 Error | Logs | `tail -50 storage/logs/laravel.log` |
| DB Error | Connection | `php artisan tinker` then `DB::connection()->getPdo()` |
| business_settings undefined | Table | `mysql ... -e "SELECT COUNT(*) FROM business_settings;"` |
| Blank page | Debug | Change `APP_DEBUG=true` in .env temporarily |
| Permission denied | Permissions | `chmod -R 775 storage/ bootstrap/cache/` |
| 404 on all pages | .htaccess | `ls -la public/.htaccess` |
| Images missing | Symlink | `php artisan storage:link` |

---

## 13. Create Support Ticket Info

If you need to contact support, provide:

```
Server Info:
- Hostname: [show with: hostname]
- PHP Version: [show with: php --version]
- MySQL Version: [show with: mysql --version]
- OS: [show with: uname -a]

Error Info:
- Full error from logs: [copy from storage/logs/laravel.log]
- When it started: [when did you first see the error?]
- What were you doing: [accessing homepage, uploading file, etc?]

Database Info:
- Database name: [from .env DB_DATABASE]
- Can connect: [yes/no, show with DB::connection()->getPdo()]
- Tables exist: [yes/no, show with: mysql ... -e "SHOW TABLES;"]
```

---

## 14. Emergency Rollback

If live is completely broken:

### Option A: Restore From Backup
```bash
# Go back to previous working codebase
cd /home/username/public_html

# If using Git
git log --oneline  # See history
git reset --hard <previous_commit_hash>

# If using backup
# Upload previous working backup
tar -xzf backup_working.tar.gz
```

### Option B: Temporary Maintenance Mode
```bash
cd /home/username/public_html

# Enable maintenance mode
php artisan down

# Fix the issue...

# Disable maintenance mode
php artisan up
```

### Option C: Test on Subdomain First
- Create subdomain: `dev.yourdomain.com`
- Deploy code there first
- Test thoroughly
- Then update main domain

---

## Summary Checklist

Run these in order to diagnose issues:

```bash
# 1. Check logs
tail -50 /home/username/public_html/storage/logs/laravel.log

# 2. Verify .env exists and has correct DB
cat /home/username/public_html/.env | grep "^DB_"

# 3. Test database
cd /home/username/public_html
php artisan tinker
> DB::connection()->getPdo()
> DB::table('business_settings')->count()
> exit

# 4. Check file permissions
ls -ld storage bootstrap/cache

# 5. Test web access
curl -I https://yourdomain.com

# 6. Clear cache
php artisan cache:clear
php artisan config:cache
php artisan view:clear
```

If all pass, your site should work!

